'use strict';

var obsidian = require('obsidian');

const defaultMaxLength = 50;
const DEFAULT_DATA = {
    recentFiles: [],
    omittedPaths: [],
    maxLength: null,
};
const RecentFilesListViewType = 'recent-files';
class RecentFilesListView extends obsidian.ItemView {
    constructor(leaf, plugin, data) {
        super(leaf);
        this.redraw = () => {
            const openFile = this.app.workspace.getActiveFile();
            const rootEl = createDiv({ cls: 'nav-folder mod-root' });
            const childrenEl = rootEl.createDiv({ cls: 'nav-folder-children' });
            this.data.recentFiles.forEach((currentFile) => {
                const navFile = childrenEl.createDiv({ cls: 'nav-file' });
                const navFileTitle = navFile.createDiv({ cls: 'nav-file-title' });
                if (openFile && currentFile.path === openFile.path) {
                    navFileTitle.addClass('is-active');
                }
                navFileTitle.createDiv({
                    cls: 'nav-file-title-content',
                    text: currentFile.basename,
                });
                navFile.setAttr('draggable', 'true');
                navFile.addEventListener('dragstart', (event) => {
                    const file = this.app.metadataCache.getFirstLinkpathDest(currentFile.path, '');
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                    const dragManager = this.app.dragManager;
                    const dragData = dragManager.dragFile(event, file);
                    dragManager.onDragStart(event, dragData);
                });
                navFile.addEventListener('mouseover', (event) => {
                    this.app.workspace.trigger('hover-link', {
                        event,
                        source: RecentFilesListViewType,
                        hoverParent: rootEl,
                        targetEl: navFile,
                        linktext: currentFile.path,
                    });
                });
                navFile.addEventListener('contextmenu', (event) => {
                    const menu = new obsidian.Menu(this.app);
                    const file = this.app.vault.getAbstractFileByPath(currentFile.path);
                    this.app.workspace.trigger('file-menu', menu, file, 'link-context-menu');
                    menu.showAtPosition({ x: event.clientX, y: event.clientY });
                });
                navFile.addEventListener('click', (event) => {
                    this.focusFile(currentFile, event.ctrlKey || event.metaKey);
                });
            });
            const contentEl = this.containerEl.children[1];
            contentEl.empty();
            contentEl.appendChild(rootEl);
        };
        this.updateData = async (file) => {
            this.data.recentFiles = this.data.recentFiles.filter((currFile) => currFile.path !== file.path);
            this.data.recentFiles.unshift({
                basename: file.basename,
                path: file.path,
            });
            await this.plugin.pruneLength(); // Handles the save
        };
        this.update = async (openedFile) => {
            if (!openedFile || !this.plugin.shouldAddFile(openedFile)) {
                return;
            }
            await this.updateData(openedFile);
            this.redraw();
        };
        /**
         * Open the provided file in the most recent leaf.
         *
         * @param shouldSplit Whether the file should be opened in a new split, or in
         * the most recent split. If the most recent split is pinned, this is set to
         * true.
         */
        this.focusFile = (file, shouldSplit = false) => {
            const targetFile = this.app.vault
                .getFiles()
                .find((f) => f.path === file.path);
            if (targetFile) {
                let leaf = this.app.workspace.getMostRecentLeaf();
                const createLeaf = shouldSplit || leaf.getViewState().pinned;
                if (createLeaf) {
                    leaf = this.app.workspace.createLeafBySplit(leaf);
                }
                leaf.openFile(targetFile);
            }
            else {
                new obsidian.Notice('Cannot find a file with that name');
                this.data.recentFiles = this.data.recentFiles.filter((fp) => fp.path !== file.path);
                this.plugin.saveData();
                this.redraw();
            }
        };
        this.plugin = plugin;
        this.data = data;
    }
    async onOpen() {
        this.redraw();
    }
    getViewType() {
        return RecentFilesListViewType;
    }
    getDisplayText() {
        return 'Recent Files';
    }
    getIcon() {
        return 'clock';
    }
    onHeaderMenu(menu) {
        menu
            .addItem((item) => {
            item
                .setTitle('Clear list')
                .setIcon('sweep')
                .onClick(async () => {
                this.data.recentFiles = [];
                await this.plugin.saveData();
                this.redraw();
            });
        })
            .addItem((item) => {
            item
                .setTitle('Close')
                .setIcon('cross')
                .onClick(() => {
                this.app.workspace.detachLeavesOfType(RecentFilesListViewType);
            });
        });
    }
    load() {
        super.load();
        this.registerEvent(this.app.workspace.on('file-open', this.update));
    }
}
class RecentFilesPlugin extends obsidian.Plugin {
    constructor() {
        super(...arguments);
        this.pruneOmittedFiles = async () => {
            this.data.recentFiles = this.data.recentFiles.filter(this.shouldAddFile);
            await this.saveData();
        };
        this.pruneLength = async () => {
            const toRemove = this.data.recentFiles.length - (this.data.maxLength || defaultMaxLength);
            if (toRemove > 0) {
                this.data.recentFiles.splice(this.data.recentFiles.length - toRemove, toRemove);
            }
            await this.saveData();
        };
        this.shouldAddFile = (file) => {
            const patterns = this.data.omittedPaths.filter((path) => path.length > 0);
            const fileMatchesRegex = (pattern) => {
                try {
                    return new RegExp(pattern).test(file.path);
                }
                catch (err) {
                    console.error('Recent Files: Invalid regex pattern: ' + pattern);
                    return false;
                }
            };
            return !patterns.some(fileMatchesRegex);
        };
        this.initView = async () => {
            let leaf = null;
            for (leaf of this.app.workspace.getLeavesOfType(RecentFilesListViewType)) {
                if (leaf.view instanceof RecentFilesListView)
                    return;
                // The view instance was created by an older version of the plugin,
                // so clear it and recreate it (so it'll be the new version).
                // This avoids the need to reload Obsidian to update the plugin.
                await leaf.setViewState({ type: 'empty' });
                break;
            }
            (leaf !== null && leaf !== void 0 ? leaf : this.app.workspace.getLeftLeaf(false)).setViewState({
                type: RecentFilesListViewType,
                active: true,
            });
        };
        this.handleRename = async (file, oldPath) => {
            const entry = this.data.recentFiles.find((recentFile) => recentFile.path === oldPath);
            if (entry) {
                entry.path = file.path;
                entry.basename = this.trimExtension(file.name);
                this.view.redraw();
                await this.saveData();
            }
        };
        this.handleDelete = async (file) => {
            const beforeLen = this.data.recentFiles.length;
            this.data.recentFiles = this.data.recentFiles.filter((recentFile) => recentFile.path !== file.path);
            if (beforeLen !== this.data.recentFiles.length) {
                this.view.redraw();
                await this.saveData();
            }
        };
        // trimExtension can be used to turn a filename into a basename when
        // interacting with a TAbstractFile that does not have a basename property.
        // private readonly trimExtension = (name: string): string => name.split('.')[0];
        // from: https://stackoverflow.com/a/4250408/617864
        this.trimExtension = (name) => name.replace(/\.[^/.]+$/, '');
    }
    async onload() {
        console.log('Recent Files: Loading plugin v' + this.manifest.version);
        await this.loadData();
        obsidian.addIcon('sweep', sweepIcon);
        this.registerView(RecentFilesListViewType, (leaf) => (this.view = new RecentFilesListView(leaf, this, this.data)));
        this.addCommand({
            id: 'recent-files-open',
            name: 'Open',
            callback: async () => {
                let [leaf] = this.app.workspace.getLeavesOfType(RecentFilesListViewType);
                if (!leaf) {
                    leaf = this.app.workspace.getLeftLeaf(false);
                    await leaf.setViewState({ type: RecentFilesListViewType });
                }
                this.app.workspace.revealLeaf(leaf);
            }
        });
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        this.app.workspace.registerHoverLinkSource(RecentFilesListViewType, {
            display: 'Recent Files',
            defaultMod: true,
        });
        if (this.app.workspace.layoutReady) {
            this.initView();
        }
        else {
            this.registerEvent(this.app.workspace.on('layout-ready', this.initView));
        }
        this.registerEvent(this.app.vault.on('rename', this.handleRename));
        this.registerEvent(this.app.vault.on('delete', this.handleDelete));
        this.addSettingTab(new RecentFilesSettingTab(this.app, this));
    }
    onunload() {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        this.app.workspace.unregisterHoverLinkSource(RecentFilesListViewType);
    }
    async loadData() {
        this.data = Object.assign(DEFAULT_DATA, await super.loadData());
        if (!this.data.maxLength) {
            console.log('Recent Files: maxLength is not set, using default (' +
                defaultMaxLength.toString() +
                ')');
        }
    }
    async saveData() {
        await super.saveData(this.data);
    }
}
class RecentFilesSettingTab extends obsidian.PluginSettingTab {
    constructor(app, plugin) {
        super(app, plugin);
        this.plugin = plugin;
    }
    display() {
        const { containerEl } = this;
        containerEl.empty();
        containerEl.createEl('h2', { text: 'Recent Files List' });
        const fragment = document.createDocumentFragment();
        const link = document.createElement('a');
        link.href =
            'https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions#writing_a_regular_expression_pattern';
        link.text = 'MDN - Regular expressions';
        fragment.append('RegExp patterns to ignore. One pattern per line. See ');
        fragment.append(link);
        fragment.append(' for help.');
        new obsidian.Setting(containerEl)
            .setName('Omitted pathname patterns')
            .setDesc(fragment)
            .addTextArea((textArea) => {
            textArea.inputEl.setAttr('rows', 6);
            textArea
                .setPlaceholder('^daily/\n\\.png$\nfoobar.*baz')
                .setValue(this.plugin.data.omittedPaths.join('\n'));
            textArea.inputEl.onblur = (e) => {
                const patterns = e.target.value;
                this.plugin.data.omittedPaths = patterns.split('\n');
                this.plugin.pruneOmittedFiles();
                this.plugin.view.redraw();
            };
        });
        new obsidian.Setting(containerEl)
            .setName('List length')
            .setDesc('Maximum number of filenames to keep in the list.')
            .addText((text) => {
            var _a;
            text.inputEl.setAttr('type', 'number');
            text.inputEl.setAttr('placeholder', defaultMaxLength);
            text
                .setValue((_a = this.plugin.data.maxLength) === null || _a === void 0 ? void 0 : _a.toString())
                .onChange((value) => {
                const parsed = parseInt(value, 10);
                if (!Number.isNaN(parsed) && parsed <= 0) {
                    new obsidian.Notice('List length must be a positive integer');
                    return;
                }
            });
            text.inputEl.onblur = (e) => {
                const maxfiles = e.target.value;
                const parsed = parseInt(maxfiles, 10);
                this.plugin.data.maxLength = parsed;
                this.plugin.pruneLength();
                this.plugin.view.redraw();
            };
        });
        const div = containerEl.createEl('div', {
            cls: 'recent-files-donation',
        });
        const donateText = document.createElement('p');
        donateText.appendText('If this plugin adds value for you and you would like to help support ' +
            'continued development, please use the buttons below:');
        div.appendChild(donateText);
        const parser = new DOMParser();
        div.appendChild(createDonateButton('https://paypal.me/tgrosinger', parser.parseFromString(paypal, 'text/xml').documentElement));
        div.appendChild(createDonateButton('https://www.buymeacoffee.com/tgrosinger', parser.parseFromString(buyMeACoffee, 'text/xml').documentElement));
    }
}
const createDonateButton = (link, img) => {
    const a = document.createElement('a');
    a.setAttribute('href', link);
    a.addClass('recent-files-donate-button');
    a.appendChild(img);
    return a;
};
const sweepIcon = `
<svg fill="currentColor" stroke="currentColor" version="1.1" viewBox="0 0 512 512" xmlns="http://www.w3.org/2000/svg">
  <path d="m495.72 1.582c-7.456-3.691-16.421-0.703-20.142 6.694l-136.92 274.08-26.818-13.433c-22.207-11.118-49.277-2.065-60.396 20.083l-6.713 13.405 160.96 80.616 6.713-13.411c11.087-22.143 2.227-49.18-20.083-60.381l-26.823-13.435 136.92-274.08c3.706-7.412 0.703-16.421-6.694-20.141z"/>
  <circle cx="173" cy="497" r="15"/>
  <circle cx="23" cy="407" r="15"/>
  <circle cx="83" cy="437" r="15"/>
  <path d="m113 482h-60c-8.276 0-15-6.724-15-15 0-8.291-6.709-15-15-15s-15 6.709-15 15c0 24.814 20.186 45 45 45h60c8.291 0 15-6.709 15-15s-6.709-15-15-15z"/>
  <path d="m108.64 388.07c-6.563 0.82-11.807 5.845-12.92 12.349-1.113 6.519 2.153 12.993 8.057 15.952l71.675 35.889c12.935 6.475 27.231 9.053 41.177 7.573-1.641 6.65 1.479 13.784 7.852 16.992l67.061 33.589c5.636 2.78 12.169 1.8 16.685-2.197 2.347-2.091 53.436-48.056 83.3-98.718l-161.6-80.94c-36.208 48.109-120.36 59.39-121.28 59.511z"/>
</svg>`;
const buyMeACoffee = `
<svg width="150" height="42" viewBox="0 0 260 73" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M0 11.68C0 5.22932 5.22931 0 11.68 0H248.2C254.651 0 259.88 5.22931 259.88 11.68V61.32C259.88 67.7707 254.651 73 248.2 73H11.68C5.22931 73 0 67.7707 0 61.32V11.68Z" fill="#FFDD00"/>
<path d="M52.2566 24.0078L52.2246 23.9889L52.1504 23.9663C52.1802 23.9915 52.2176 24.0061 52.2566 24.0078Z" fill="#0D0C22"/>
<path d="M52.7248 27.3457L52.6895 27.3556L52.7248 27.3457Z" fill="#0D0C22"/>
<path d="M52.2701 24.0024C52.266 24.0019 52.2619 24.0009 52.258 23.9995C52.2578 24.0022 52.2578 24.0049 52.258 24.0076C52.2624 24.007 52.2666 24.0052 52.2701 24.0024Z" fill="#0D0C22"/>
<path d="M52.2578 24.0094H52.2643V24.0054L52.2578 24.0094Z" fill="#0D0C22"/>
<path d="M52.6973 27.3394L52.7513 27.3086L52.7714 27.2973L52.7897 27.2778C52.7554 27.2926 52.7241 27.3135 52.6973 27.3394Z" fill="#0D0C22"/>
<path d="M52.3484 24.0812L52.2956 24.031L52.2598 24.0115C52.279 24.0454 52.3108 24.0705 52.3484 24.0812Z" fill="#0D0C22"/>
<path d="M39.0684 56.469C39.0262 56.4872 38.9893 56.5158 38.9609 56.552L38.9943 56.5306C39.0169 56.5098 39.0489 56.4853 39.0684 56.469Z" fill="#0D0C22"/>
<path d="M46.7802 54.9518C46.7802 54.9041 46.7569 54.9129 46.7626 55.0826C46.7626 55.0687 46.7683 55.0549 46.7708 55.0417C46.7739 55.0115 46.7764 54.982 46.7802 54.9518Z" fill="#0D0C22"/>
<path d="M45.9844 56.469C45.9422 56.4872 45.9053 56.5158 45.877 56.552L45.9103 56.5306C45.9329 56.5098 45.9649 56.4853 45.9844 56.469Z" fill="#0D0C22"/>
<path d="M33.6307 56.8301C33.5987 56.8023 33.5595 56.784 33.5176 56.7773C33.5515 56.7937 33.5855 56.81 33.6081 56.8226L33.6307 56.8301Z" fill="#0D0C22"/>
<path d="M32.4118 55.6598C32.4068 55.6103 32.3916 55.5624 32.3672 55.519C32.3845 55.5642 32.399 55.6104 32.4106 55.6573L32.4118 55.6598Z" fill="#0D0C22"/>
<path d="M40.623 34.7221C38.9449 35.4405 37.0404 36.2551 34.5722 36.2551C33.5397 36.2531 32.5122 36.1114 31.5176 35.834L33.2247 53.3605C33.2851 54.093 33.6188 54.7761 34.1595 55.2739C34.7003 55.7718 35.4085 56.0482 36.1435 56.048C36.1435 56.048 38.564 56.1737 39.3716 56.1737C40.2409 56.1737 42.8474 56.048 42.8474 56.048C43.5823 56.048 44.2904 55.7716 44.831 55.2737C45.3716 54.7759 45.7052 54.0929 45.7656 53.3605L47.594 33.993C46.7769 33.714 45.9523 33.5286 45.0227 33.5286C43.415 33.5279 42.1196 34.0817 40.623 34.7221Z" fill="white"/>
<path d="M26.2344 27.2449L26.2633 27.2719L26.2821 27.2832C26.2676 27.2688 26.2516 27.2559 26.2344 27.2449Z" fill="#0D0C22"/>
<path d="M55.4906 25.6274L55.2336 24.3307C55.0029 23.1673 54.4793 22.068 53.2851 21.6475C52.9024 21.513 52.468 21.4552 52.1745 21.1768C51.881 20.8983 51.7943 20.4659 51.7264 20.0649C51.6007 19.3289 51.4825 18.5923 51.3537 17.8575C51.2424 17.2259 51.1544 16.5163 50.8647 15.9368C50.4876 15.1586 49.705 14.7036 48.9269 14.4025C48.5282 14.2537 48.1213 14.1278 47.7082 14.0254C45.7642 13.5125 43.7202 13.324 41.7202 13.2165C39.3197 13.084 36.9128 13.1239 34.518 13.3359C32.7355 13.4981 30.8581 13.6942 29.1642 14.3108C28.5451 14.5364 27.9071 14.8073 27.4364 15.2856C26.8587 15.8733 26.6702 16.7821 27.0919 17.515C27.3917 18.0354 27.8996 18.4031 28.4382 18.6463C29.1398 18.9597 29.8726 19.1982 30.6242 19.3578C32.7172 19.8204 34.885 20.0021 37.0233 20.0794C39.3932 20.175 41.767 20.0975 44.1256 19.8474C44.7089 19.7833 45.2911 19.7064 45.8723 19.6168C46.5568 19.5118 46.9961 18.6168 46.7943 17.9933C46.553 17.2479 45.9044 16.9587 45.1709 17.0712C45.0628 17.0882 44.9553 17.1039 44.8472 17.1196L44.7692 17.131C44.5208 17.1624 44.2723 17.1917 44.0238 17.219C43.5105 17.2743 42.9959 17.3195 42.4801 17.3547C41.3249 17.4352 40.1665 17.4722 39.0088 17.4741C37.8712 17.4741 36.7329 17.4421 35.5978 17.3673C35.0799 17.3333 34.5632 17.2902 34.0478 17.2378C33.8134 17.2133 33.5796 17.1875 33.3458 17.1586L33.1233 17.1303L33.0749 17.1234L32.8442 17.0901C32.3728 17.0191 31.9014 16.9374 31.435 16.8387C31.388 16.8283 31.3459 16.8021 31.3157 16.7645C31.2856 16.7269 31.2691 16.6801 31.2691 16.6319C31.2691 16.5837 31.2856 16.5369 31.3157 16.4993C31.3459 16.4617 31.388 16.4356 31.435 16.4251H31.4438C31.848 16.339 32.2553 16.2655 32.6638 16.2014C32.8 16.18 32.9366 16.159 33.0736 16.1385H33.0774C33.3332 16.1215 33.5903 16.0757 33.8448 16.0455C36.0595 15.8151 38.2874 15.7366 40.5128 15.8104C41.5933 15.8419 42.6731 15.9053 43.7485 16.0147C43.9798 16.0386 44.2098 16.0637 44.4399 16.092C44.5279 16.1027 44.6165 16.1153 44.7051 16.1259L44.8836 16.1517C45.404 16.2292 45.9217 16.3233 46.4367 16.4339C47.1997 16.5999 48.1796 16.6539 48.519 17.4898C48.6271 17.7551 48.6761 18.0499 48.7359 18.3283L48.8119 18.6834C48.8139 18.6898 48.8154 18.6963 48.8163 18.7029C48.9961 19.5409 49.176 20.379 49.3562 21.217C49.3694 21.2789 49.3697 21.3429 49.3571 21.4049C49.3445 21.4669 49.3193 21.5257 49.2829 21.5776C49.2466 21.6294 49.2 21.6732 49.146 21.7062C49.092 21.7392 49.0317 21.7608 48.969 21.7695H48.964L48.854 21.7846L48.7453 21.799C48.4009 21.8439 48.056 21.8858 47.7107 21.9247C47.0307 22.0022 46.3496 22.0693 45.6674 22.1259C44.3119 22.2386 42.9536 22.3125 41.5927 22.3477C40.8992 22.3662 40.2059 22.3748 39.5129 22.3735C36.7543 22.3713 33.9981 22.211 31.2578 21.8933C30.9611 21.8581 30.6645 21.8204 30.3678 21.7821C30.5978 21.8116 30.2006 21.7594 30.1202 21.7481C29.9316 21.7217 29.7431 21.6943 29.5545 21.6658C28.9216 21.5709 28.2924 21.454 27.6607 21.3515C26.8971 21.2258 26.1667 21.2887 25.476 21.6658C24.909 21.976 24.4501 22.4518 24.1605 23.0297C23.8626 23.6456 23.7739 24.3163 23.6407 24.9781C23.5074 25.6399 23.3 26.3521 23.3786 27.0315C23.5477 28.4979 24.5728 29.6895 26.0473 29.956C27.4345 30.2074 28.8292 30.4111 30.2276 30.5846C35.7212 31.2574 41.2711 31.3379 46.7818 30.8247C47.2305 30.7828 47.6787 30.7371 48.1262 30.6876C48.266 30.6723 48.4074 30.6884 48.5401 30.7348C48.6729 30.7812 48.7936 30.8566 48.8934 30.9557C48.9932 31.0548 49.0695 31.1749 49.1169 31.3073C49.1642 31.4397 49.1814 31.5811 49.167 31.7209L49.0275 33.0773C48.7463 35.8181 48.4652 38.5587 48.184 41.299C47.8907 44.1769 47.5955 47.0545 47.2984 49.9319C47.2146 50.7422 47.1308 51.5524 47.047 52.3624C46.9666 53.16 46.9552 53.9827 46.8038 54.7709C46.5649 56.0103 45.7258 56.7715 44.5015 57.0499C43.3798 57.3052 42.2339 57.4392 41.0836 57.4497C39.8083 57.4566 38.5336 57.4 37.2583 57.4069C35.897 57.4145 34.2295 57.2887 33.1786 56.2756C32.2553 55.3856 32.1277 53.9921 32.002 52.7872C31.8344 51.192 31.6682 49.5971 31.5036 48.0023L30.5796 39.1344L29.9819 33.3966C29.9718 33.3017 29.9618 33.208 29.9524 33.1125C29.8807 32.428 29.3961 31.758 28.6324 31.7926C27.9788 31.8215 27.2359 32.3771 27.3125 33.1125L27.7557 37.3664L28.672 46.1657C28.9331 48.6652 29.1935 51.165 29.4533 53.6653C29.5036 54.1442 29.5507 54.6244 29.6035 55.1034C29.8908 57.7205 31.8895 59.131 34.3646 59.5282C35.8102 59.7607 37.291 59.8085 38.758 59.8324C40.6386 59.8626 42.538 59.9348 44.3877 59.5942C47.1287 59.0914 49.1853 57.2611 49.4788 54.422C49.5626 53.6024 49.6464 52.7826 49.7302 51.9626C50.0088 49.2507 50.2871 46.5386 50.5649 43.8263L51.4737 34.9641L51.8904 30.9026C51.9112 30.7012 51.9962 30.5118 52.133 30.3625C52.2697 30.2132 52.4509 30.1119 52.6497 30.0736C53.4335 29.9208 54.1827 29.66 54.7402 29.0635C55.6277 28.1138 55.8043 26.8756 55.4906 25.6274ZM26.0071 26.5035C26.019 26.4979 25.997 26.6003 25.9876 26.6481C25.9857 26.5758 25.9895 26.5117 26.0071 26.5035ZM26.0831 27.0918C26.0894 27.0874 26.1083 27.1126 26.1278 27.1428C26.0982 27.1151 26.0794 27.0944 26.0825 27.0918H26.0831ZM26.1579 27.1905C26.185 27.2364 26.1994 27.2653 26.1579 27.1905V27.1905ZM26.3082 27.3125H26.3119C26.3119 27.3169 26.3188 27.3213 26.3214 27.3257C26.3172 27.3208 26.3126 27.3164 26.3075 27.3125H26.3082ZM52.6132 27.1302C52.3317 27.3979 51.9074 27.5224 51.4882 27.5846C46.7868 28.2823 42.0169 28.6355 37.264 28.4796C33.8624 28.3633 30.4967 27.9856 27.129 27.5098C26.799 27.4633 26.4414 27.403 26.2145 27.1597C25.7871 26.7009 25.997 25.777 26.1083 25.2226C26.2101 24.7148 26.405 24.0378 27.009 23.9656C27.9518 23.8549 29.0466 24.2528 29.9794 24.3942C31.1023 24.5656 32.2295 24.7028 33.3609 24.8059C38.1892 25.2459 43.0986 25.1774 47.9056 24.5337C48.7817 24.416 49.6548 24.2792 50.5246 24.1233C51.2996 23.9844 52.1588 23.7236 52.6271 24.5262C52.9482 25.073 52.991 25.8046 52.9413 26.4225C52.926 26.6917 52.8084 26.9448 52.6126 27.1302H52.6132Z" fill="#0D0C22"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M81.1302 40.1929C80.8556 40.7169 80.4781 41.1732 79.9978 41.5604C79.5175 41.9479 78.9571 42.2633 78.3166 42.5062C77.6761 42.7497 77.0315 42.9131 76.3835 42.9964C75.7352 43.0799 75.106 43.0727 74.4963 42.9735C73.8863 42.8749 73.3674 42.6737 72.9408 42.3695L73.4214 37.3779C73.8633 37.2261 74.4197 37.0703 75.0909 36.9107C75.7619 36.7513 76.452 36.6371 77.1613 36.5689C77.8705 36.5003 78.5412 36.5084 79.1744 36.5917C79.8068 36.6753 80.3065 36.8765 80.6725 37.1958C80.8707 37.378 81.0387 37.5754 81.176 37.7883C81.313 38.0011 81.3969 38.2214 81.4276 38.4493C81.5037 39.0875 81.4047 39.6687 81.1302 40.1929ZM74.153 29.5602C74.4734 29.3627 74.8585 29.1877 75.3083 29.0356C75.7581 28.8841 76.2195 28.7774 76.6923 28.7167C77.1648 28.6562 77.6262 28.6481 78.0763 28.6938C78.5258 28.7395 78.9228 28.8647 79.2659 29.0697C79.6089 29.2751 79.8643 29.5714 80.032 29.9586C80.1997 30.3464 80.2456 30.8365 80.1693 31.429C80.1083 31.9001 79.9211 32.2991 79.6089 32.6256C79.2963 32.9526 78.9147 33.2259 78.4652 33.4462C78.0154 33.6668 77.5388 33.8415 77.0356 33.9702C76.5321 34.0997 76.0477 34.1949 75.5828 34.2553C75.1176 34.3163 74.7137 34.3545 74.3706 34.3692C74.0273 34.3845 73.8021 34.3921 73.6956 34.3921L74.153 29.5602ZM83.6007 36.9676C83.3566 36.4361 83.0287 35.9689 82.6172 35.5658C82.2054 35.1633 81.717 34.8709 81.1531 34.6885C81.3969 34.491 81.6371 34.1795 81.8737 33.7539C82.1099 33.3288 82.3119 32.865 82.4796 32.3636C82.6474 31.8619 82.762 31.357 82.8229 30.8478C82.8836 30.3389 82.8607 29.902 82.7544 29.537C82.4947 28.6256 82.087 27.9114 81.5303 27.3946C80.9734 26.8782 80.3257 26.5211 79.586 26.3233C78.8462 26.1264 78.0304 26.0842 77.1383 26.1981C76.2462 26.312 75.3347 26.5361 74.4049 26.8704C74.4049 26.7946 74.4124 26.7148 74.4278 26.6312C74.4426 26.548 74.4504 26.4604 74.4504 26.369C74.4504 26.1411 74.3361 25.9439 74.1074 25.7765C73.8787 25.6093 73.6155 25.5107 73.3183 25.4801C73.0209 25.45 72.731 25.5142 72.4489 25.6738C72.1665 25.8334 71.9721 26.1264 71.8656 26.5511C71.7434 27.9189 71.6215 29.3398 71.4996 30.8134C71.3774 32.2875 71.248 33.7767 71.1107 35.2812C70.9735 36.7855 70.8362 38.2784 70.6989 39.7598C70.5616 41.2414 70.4244 42.6659 70.2871 44.0333C70.333 44.4436 70.4473 44.7629 70.6304 44.9907C70.8133 45.2189 71.0268 45.3556 71.2709 45.401C71.5147 45.4467 71.7704 45.4045 72.0371 45.2755C72.3038 45.1469 72.5365 44.9222 72.735 44.6032C73.3447 44.9375 74.0311 45.1541 74.7938 45.253C75.5561 45.3516 76.3298 45.3516 77.1157 45.253C77.9007 45.1541 78.6747 44.9682 79.4374 44.6943C80.1997 44.4211 80.8936 44.079 81.519 43.669C82.1441 43.2586 82.6703 42.7911 83.0975 42.2671C83.5244 41.7426 83.8065 41.1767 83.9437 40.5691C84.081 39.946 84.119 39.3231 84.0581 38.7C83.9971 38.0771 83.8445 37.5 83.6007 36.9676Z" fill="#0D0C23"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M105.915 49.0017C105.832 49.5031 105.713 50.0311 105.561 50.586C105.408 51.1403 105.229 51.6458 105.023 52.1018C104.818 52.5575 104.589 52.9256 104.337 53.207C104.085 53.488 103.815 53.606 103.525 53.5606C103.296 53.5297 103.151 53.3854 103.091 53.1274C103.029 52.8686 103.029 52.5497 103.091 52.17C103.151 51.7901 103.269 51.3607 103.445 50.8821C103.62 50.4035 103.834 49.9284 104.085 49.4577C104.337 48.9864 104.623 48.5347 104.943 48.1015C105.264 47.6686 105.599 47.3075 105.95 47.0189C106.026 47.11 106.06 47.3378 106.053 47.7028C106.045 48.0674 105.999 48.5006 105.915 49.0017ZM113.67 39.1097C113.464 38.8819 113.213 38.7529 112.915 38.7223C112.618 38.6919 112.317 38.859 112.012 39.2237C111.813 39.5883 111.562 39.9379 111.257 40.2722C110.952 40.6067 110.635 40.9103 110.307 41.1839C109.98 41.4572 109.667 41.6931 109.37 41.8903C109.072 42.0881 108.84 42.2324 108.672 42.3235C108.611 41.8374 108.576 41.3132 108.569 40.7507C108.561 40.1886 108.573 39.619 108.603 39.0415C108.649 38.2209 108.744 37.393 108.889 36.557C109.034 35.7213 109.244 34.9007 109.518 34.0951C109.518 33.67 109.419 33.3242 109.221 33.0582C109.022 32.7924 108.782 32.625 108.5 32.5567C108.218 32.4885 107.929 32.5264 107.631 32.6707C107.334 32.8153 107.078 33.0775 106.865 33.4569C106.682 33.9586 106.472 34.5207 106.236 35.1436C105.999 35.7667 105.732 36.4012 105.435 37.0469C105.138 37.6931 104.806 38.3197 104.44 38.9273C104.074 39.5354 103.674 40.075 103.239 40.5457C102.804 41.0168 102.331 41.3854 101.821 41.6512C101.31 41.9172 100.757 42.0349 100.162 42.0045C99.8876 41.9285 99.6893 41.7235 99.5675 41.3889C99.4453 41.0549 99.373 40.6368 99.3504 40.1354C99.3275 39.634 99.3504 39.0831 99.4189 38.4828C99.4877 37.8828 99.5791 37.2863 99.6934 36.6938C99.8078 36.101 99.9337 35.5389 100.071 35.0071C100.208 34.4753 100.337 34.0268 100.46 33.6622C100.643 33.2218 100.643 32.8529 100.46 32.5567C100.277 32.2604 100.025 32.0631 99.705 31.964C99.3846 31.8654 99.0489 31.8694 98.6983 31.9755C98.3474 32.0819 98.0958 32.3173 97.9435 32.682C97.684 33.3054 97.4475 34.004 97.2342 34.779C97.0206 35.5539 96.8491 36.3558 96.7197 37.1836C96.5896 38.0121 96.5171 38.8327 96.502 39.6456C96.5011 39.6985 96.5037 39.7488 96.5034 39.8014C96.1709 40.6848 95.854 41.3525 95.553 41.7992C95.1641 42.377 94.7253 42.6277 94.2375 42.5513C94.0236 42.4603 93.8832 42.2477 93.8147 41.9132C93.7453 41.5792 93.7227 41.1689 93.7453 40.6822C93.7688 40.1964 93.826 39.6456 93.9171 39.0299C94.0091 38.4146 94.1229 37.7764 94.2601 37.1154C94.3977 36.4541 94.5425 35.7899 94.6949 35.121C94.8472 34.4525 94.9845 33.8218 95.107 33.2291C95.0916 32.6973 94.9352 32.291 94.6377 32.0097C94.3405 31.7289 93.9247 31.6187 93.3913 31.6791C93.0253 31.8312 92.7542 32.029 92.579 32.2719C92.4034 32.5148 92.2623 32.8265 92.1558 33.2062C92.0946 33.404 92.0032 33.799 91.8813 34.3918C91.7591 34.984 91.603 35.6644 91.4123 36.4315C91.2217 37.1992 90.9967 38.0005 90.7376 38.8362C90.4781 39.6719 90.1885 40.4283 89.8684 41.1041C89.548 41.7801 89.1972 42.3235 88.8161 42.7338C88.4348 43.1438 88.023 43.3113 87.5807 43.2352C87.3366 43.1895 87.1805 42.9388 87.112 42.4831C87.0432 42.0271 87.0319 41.4653 87.0775 40.7964C87.1233 40.1279 87.2148 39.3946 87.352 38.5971C87.4893 37.7993 87.63 37.0434 87.7752 36.3289C87.92 35.6149 88.0535 34.984 88.1756 34.4372C88.2975 33.8901 88.3814 33.5254 88.4272 33.3433C88.4272 32.9026 88.3277 32.5495 88.1298 32.2832C87.9313 32.0178 87.6913 31.8503 87.4092 31.7818C87.1268 31.7136 86.8372 31.7514 86.54 31.8957C86.2426 32.0403 85.9872 32.3026 85.7736 32.682C85.6973 33.0923 85.598 33.5674 85.4761 34.1067C85.3539 34.6459 85.2361 35.2006 85.1218 35.7705C85.0074 36.3404 84.9003 36.8988 84.8014 37.4459C84.7021 37.993 84.6299 38.4716 84.584 38.8819C84.5536 39.2008 84.519 39.5923 84.4813 40.0556C84.443 40.5194 84.4238 41.0092 84.4238 41.5257C84.4238 42.0427 84.4618 42.5554 84.5385 43.0643C84.6145 43.5735 84.7518 44.0408 84.95 44.4659C85.1482 44.8915 85.4265 45.2408 85.7852 45.5144C86.1433 45.7879 86.5972 45.9397 87.1463 45.9704C87.7101 46.0005 88.202 45.9591 88.6217 45.8449C89.041 45.731 89.4221 45.5523 89.7654 45.3091C90.1084 45.0665 90.421 44.7776 90.7033 44.443C90.9851 44.1091 91.2637 43.7444 91.5383 43.3491C91.7974 43.9269 92.1329 44.3748 92.5447 44.694C92.9565 45.013 93.3913 45.2032 93.8486 45.2637C94.306 45.3241 94.7715 45.2602 95.2442 45.0699C95.7167 44.8803 96.1436 44.5573 96.5252 44.1012C96.7762 43.8216 97.0131 43.5038 97.2354 43.1525C97.3297 43.317 97.4301 43.4758 97.543 43.6224C97.9168 44.1091 98.424 44.443 99.0645 44.6255C99.7506 44.808 100.421 44.8386 101.077 44.7169C101.733 44.5954 102.358 44.3748 102.953 44.0559C103.548 43.7366 104.101 43.3532 104.612 42.9047C105.122 42.4565 105.568 41.9895 105.95 41.5028C105.934 41.8524 105.927 42.1832 105.927 42.4944C105.927 42.8061 105.919 43.1438 105.904 43.5088C105.141 44.0408 104.421 44.679 103.742 45.4233C103.064 46.1676 102.469 46.9616 101.958 47.8051C101.447 48.6483 101.047 49.5031 100.757 50.3691C100.467 51.2357 100.326 52.0445 100.334 52.7969C100.341 53.549 100.521 54.206 100.871 54.7681C101.222 55.3306 101.794 55.7331 102.587 55.9763C103.411 56.2348 104.135 56.242 104.76 55.9991C105.386 55.7559 105.931 55.3531 106.396 54.791C106.861 54.2289 107.242 53.549 107.54 52.7512C107.837 51.9534 108.073 51.1215 108.249 50.2555C108.424 49.3894 108.535 48.5379 108.58 47.7028C108.626 46.8668 108.626 46.1219 108.58 45.4687C109.892 44.9219 110.967 44.2305 111.806 43.3945C112.645 42.5594 113.338 41.6778 113.887 40.7507C114.055 40.5229 114.112 40.2493 114.059 39.9304C114.006 39.6111 113.876 39.3376 113.67 39.1097Z" fill="#0D0C23"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M142.53 37.6515C142.575 37.3022 142.644 36.9335 142.735 36.546C142.827 36.1585 142.941 35.7823 143.079 35.4177C143.216 35.0531 143.376 34.7379 143.559 34.4718C143.742 34.2061 143.937 34.0161 144.142 33.9019C144.348 33.7883 144.558 33.7995 144.771 33.936C145 34.0731 145.141 34.3617 145.195 34.8021C145.248 35.2433 145.195 35.7141 145.034 36.2155C144.874 36.7172 144.588 37.1879 144.177 37.6286C143.765 38.0696 143.208 38.3579 142.507 38.4947C142.476 38.2824 142.484 38.0011 142.53 37.6515ZM150.456 38.5857C150.204 38.5103 149.964 38.5025 149.735 38.5632C149.506 38.6239 149.361 38.7835 149.301 39.042C149.178 39.5281 148.984 40.0258 148.717 40.5347C148.45 41.0439 148.122 41.5262 147.734 41.9822C147.345 42.438 146.906 42.8408 146.418 43.1901C145.93 43.5397 145.419 43.7904 144.886 43.9422C144.351 44.1096 143.91 44.1284 143.559 43.9991C143.208 43.8705 142.93 43.6498 142.724 43.3384C142.518 43.027 142.369 42.6508 142.278 42.2101C142.186 41.7694 142.133 41.3137 142.118 40.8424C142.987 40.9034 143.761 40.7478 144.44 40.3751C145.118 40.0032 145.694 39.509 146.167 38.8937C146.639 38.2784 146.998 37.587 147.242 36.8195C147.485 36.0524 147.623 35.2887 147.653 34.5288C147.669 33.8146 147.562 33.2108 147.333 32.7169C147.105 32.2233 146.796 31.839 146.407 31.5658C146.018 31.2922 145.572 31.1326 145.069 31.0872C144.566 31.0415 144.054 31.11 143.536 31.2922C142.91 31.505 142.381 31.8506 141.946 32.3294C141.512 32.808 141.149 33.3629 140.86 33.9933C140.57 34.6239 140.341 35.3038 140.173 36.033C140.005 36.7626 139.883 37.4806 139.807 38.1873C139.739 38.8214 139.702 39.4278 139.689 40.013C139.657 40.0874 139.625 40.1588 139.59 40.2383C139.354 40.7782 139.079 41.3062 138.766 41.8226C138.454 42.3394 138.107 42.7725 137.726 43.1218C137.344 43.4714 136.948 43.5929 136.536 43.4865C136.292 43.426 136.159 43.1444 136.136 42.6433C136.113 42.1416 136.139 41.5187 136.216 40.7741C136.292 40.0298 136.38 39.2239 136.479 38.3579C136.578 37.4918 136.628 36.664 136.628 35.8737C136.628 35.1898 136.498 34.5329 136.239 33.9019C135.979 33.2718 135.625 32.7473 135.175 32.3294C134.725 31.9113 134.203 31.634 133.608 31.4975C133.013 31.3605 132.373 31.4518 131.687 31.7708C131 32.09 130.455 32.5382 130.051 33.1157C129.647 33.6934 129.277 34.3009 128.942 34.9391C128.819 34.4528 128.641 34.0011 128.404 33.583C128.167 33.1651 127.878 32.8005 127.535 32.4888C127.191 32.1776 126.806 31.9344 126.38 31.7595C125.953 31.5851 125.502 31.4975 125.03 31.4975C124.572 31.4975 124.149 31.5851 123.76 31.7595C123.371 31.9344 123.017 32.1583 122.696 32.4318C122.376 32.7056 122.087 33.013 121.827 33.3551C121.568 33.6969 121.339 34.0352 121.141 34.3692C121.11 33.9742 121.076 33.6286 121.038 33.332C121 33.0359 120.931 32.7852 120.832 32.5801C120.733 32.3748 120.592 32.2193 120.409 32.1129C120.226 32.0067 119.967 31.9532 119.632 31.9532C119.464 31.9532 119.296 31.9874 119.128 32.0556C118.96 32.1241 118.811 32.2193 118.682 32.3407C118.552 32.4627 118.453 32.6105 118.385 32.7852C118.316 32.9598 118.297 33.1614 118.327 33.3892C118.342 33.5566 118.385 33.7576 118.453 33.9933C118.522 34.2289 118.587 34.5369 118.648 34.9163C118.708 35.2962 118.758 35.756 118.796 36.2953C118.834 36.8349 118.846 37.4959 118.831 38.2784C118.815 39.0611 118.758 39.9763 118.659 41.0248C118.56 42.0733 118.403 43.289 118.19 44.6714C118.16 44.9907 118.282 45.2492 118.556 45.4467C118.831 45.6439 119.143 45.7578 119.494 45.7885C119.845 45.8188 120.177 45.7578 120.489 45.6063C120.802 45.4539 120.981 45.1882 121.027 44.8085C121.072 44.0943 121.16 43.3347 121.29 42.529C121.419 41.724 121.579 40.9262 121.77 40.1359C121.961 39.346 122.178 38.5938 122.422 37.8793C122.666 37.1651 122.937 36.5347 123.234 35.9876C123.532 35.4405 123.84 35.0039 124.161 34.6771C124.481 34.3504 124.816 34.187 125.167 34.187C125.594 34.187 125.926 34.3805 126.162 34.7679C126.398 35.1557 126.566 35.6536 126.666 36.2609C126.765 36.869 126.81 37.5341 126.803 38.2555C126.795 38.9773 126.765 39.6724 126.711 40.341C126.658 41.0098 126.597 41.606 126.528 42.1303C126.46 42.6545 126.41 43.0157 126.38 43.2129C126.38 43.5625 126.513 43.8395 126.78 44.0448C127.046 44.2498 127.344 44.3716 127.672 44.4095C128 44.4476 128.309 44.3866 128.598 44.227C128.888 44.0674 129.056 43.7982 129.102 43.4179C129.254 42.324 129.464 41.2264 129.731 40.1247C129.997 39.023 130.303 38.0355 130.646 37.1616C130.989 36.2878 131.37 35.5735 131.79 35.0189C132.209 34.4646 132.655 34.187 133.128 34.187C133.371 34.187 133.559 34.3544 133.688 34.6884C133.818 35.0227 133.883 35.4784 133.883 36.0559C133.883 36.4815 133.848 36.9184 133.78 37.3666C133.711 37.8148 133.631 38.2784 133.54 38.7569C133.448 39.2358 133.368 39.7256 133.299 40.227C133.231 40.7287 133.196 41.2527 133.196 41.7998C133.196 42.1797 133.235 42.6204 133.311 43.1218C133.387 43.6229 133.532 44.0983 133.745 44.5462C133.959 44.9947 134.252 45.3744 134.626 45.6858C135 45.9973 135.476 46.1531 136.056 46.1531C136.925 46.1531 137.695 45.9669 138.366 45.5947C139.037 45.2226 139.613 44.7365 140.093 44.1362C140.118 44.1047 140.141 44.0711 140.165 44.0399C140.202 44.1287 140.235 44.2227 140.276 44.3071C140.604 44.9756 141.05 45.4921 141.615 45.857C142.178 46.2216 142.842 46.4229 143.605 46.4611C144.367 46.4987 145.198 46.3581 146.098 46.0392C146.769 45.796 147.352 45.4921 147.848 45.1275C148.343 44.7628 148.789 44.3184 149.186 43.7941C149.583 43.2699 149.945 42.6658 150.273 41.9822C150.601 41.2981 150.932 40.5159 151.268 39.6342C151.329 39.3916 151.272 39.1751 151.097 38.9848C150.921 38.7951 150.708 38.6621 150.456 38.5857Z" fill="#0D0C23"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M162.887 36.0434C162.81 36.4918 162.707 36.986 162.578 37.525C162.448 38.0646 162.284 38.623 162.086 39.2004C161.888 39.7779 161.644 40.2984 161.354 40.7616C161.064 41.2254 160.733 41.5935 160.359 41.8671C159.985 42.1406 159.555 42.2546 159.066 42.2089C158.822 42.1788 158.635 42.0117 158.506 41.7075C158.376 41.4038 158.308 41.0161 158.3 40.545C158.292 40.0743 158.334 39.5575 158.426 38.9951C158.517 38.4333 158.658 37.8821 158.849 37.3426C159.04 36.8036 159.272 36.3056 159.547 35.8496C159.821 35.3939 160.138 35.0405 160.496 34.7898C160.854 34.5391 161.247 34.4217 161.674 34.4365C162.101 34.4518 162.559 34.6643 163.047 35.0747C163.016 35.2725 162.963 35.5954 162.887 36.0434ZM171.019 37.787C170.782 37.6656 170.538 37.6392 170.287 37.7075C170.035 37.7757 169.856 38.0076 169.749 38.4026C169.688 38.8283 169.551 39.3294 169.338 39.9069C169.124 40.4843 168.861 41.0317 168.548 41.5478C168.236 42.0646 167.877 42.494 167.473 42.8358C167.069 43.1778 166.638 43.3337 166.181 43.3028C165.799 43.2727 165.532 43.079 165.38 42.7218C165.227 42.3647 165.147 41.9168 165.14 41.3769C165.132 40.838 165.186 40.2301 165.3 39.5538C165.414 38.8777 165.552 38.2054 165.712 37.5363C165.872 36.868 166.036 36.2258 166.204 35.6105C166.371 34.9951 166.508 34.4747 166.616 34.0493C166.738 33.6693 166.699 33.3466 166.501 33.0803C166.303 32.8149 166.055 32.6246 165.758 32.5107C165.46 32.3967 165.159 32.3664 164.854 32.4196C164.549 32.4728 164.351 32.6362 164.259 32.9094C163.359 32.1345 162.494 31.7166 161.663 31.6559C160.831 31.5952 160.065 31.7776 159.364 32.203C158.662 32.6284 158.041 33.2437 157.5 34.0493C156.958 34.8549 156.52 35.7322 156.184 36.6818C155.849 37.6314 155.639 38.6004 155.555 39.5879C155.471 40.5757 155.536 41.4761 155.75 42.289C155.963 43.1018 156.34 43.7669 156.882 44.283C157.423 44.7998 158.159 45.0583 159.089 45.0583C159.501 45.0583 159.898 44.9747 160.279 44.8076C160.66 44.6401 161.011 44.4426 161.331 44.2148C161.651 43.9869 161.933 43.7475 162.178 43.4968C162.421 43.2461 162.612 43.0373 162.749 42.8699C162.856 43.417 163.032 43.8808 163.276 44.2605C163.519 44.6401 163.798 44.9521 164.111 45.1948C164.423 45.4376 164.751 45.6164 165.094 45.7306C165.437 45.8445 165.769 45.9015 166.089 45.9015C166.806 45.9015 167.477 45.6583 168.102 45.1719C168.727 44.6861 169.288 44.0893 169.784 43.3829C170.279 42.6762 170.687 41.9319 171.007 41.1491C171.328 40.3666 171.541 39.6715 171.648 39.0634C171.755 38.8355 171.735 38.5964 171.591 38.3457C171.446 38.095 171.255 37.909 171.019 37.787Z" fill="#0D0C23"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M212.194 50.3701C212.064 50.8866 211.862 51.3238 211.587 51.6806C211.313 52.0377 210.97 52.2239 210.558 52.2393C210.299 52.2543 210.101 52.1175 209.963 51.8289C209.826 51.5401 209.731 51.1679 209.678 50.7122C209.624 50.2562 209.601 49.747 209.609 49.1849C209.616 48.6227 209.639 48.0681 209.678 47.521C209.715 46.9742 209.761 46.4647 209.815 45.9939C209.868 45.5226 209.91 45.1586 209.94 44.9C210.459 44.9608 210.89 45.1846 211.233 45.5723C211.576 45.9598 211.839 46.4193 212.022 46.9514C212.205 47.4831 212.312 48.0568 212.343 48.6722C212.373 49.2875 212.323 49.8534 212.194 50.3701ZM203.913 50.3701C203.783 50.8866 203.581 51.3238 203.307 51.6806C203.032 52.0377 202.689 52.2239 202.277 52.2393C202.018 52.2543 201.82 52.1175 201.683 51.8289C201.545 51.5401 201.45 51.1679 201.397 50.7122C201.343 50.2562 201.32 49.747 201.328 49.1849C201.336 48.6227 201.358 48.0681 201.397 47.521C201.434 46.9742 201.48 46.4647 201.534 45.9939C201.587 45.5226 201.629 45.1586 201.66 44.9C202.178 44.9608 202.609 45.1846 202.952 45.5723C203.295 45.9598 203.558 46.4193 203.741 46.9514C203.924 47.4831 204.031 48.0568 204.062 48.6722C204.092 49.2875 204.042 49.8534 203.913 50.3701ZM195.415 37.4241C195.399 37.7884 195.365 38.1114 195.312 38.3925C195.258 38.6741 195.186 38.8522 195.095 38.9283C194.927 38.8369 194.721 38.6018 194.477 38.2216C194.233 37.8419 194.042 37.4122 193.905 36.9336C193.768 36.4551 193.725 35.9843 193.779 35.5205C193.832 35.0573 194.073 34.6967 194.5 34.4379C194.667 34.3468 194.812 34.3809 194.934 34.5405C195.056 34.7001 195.155 34.9318 195.232 35.2357C195.308 35.5399 195.361 35.8892 195.392 36.2842C195.422 36.6795 195.43 37.0591 195.415 37.4241ZM193.39 41.9711C193.154 42.2215 192.89 42.4381 192.601 42.6206C192.311 42.803 192.014 42.9398 191.709 43.0309C191.404 43.1223 191.129 43.1448 190.885 43.0991C190.199 42.9627 189.673 42.666 189.307 42.2103C188.941 41.7545 188.708 41.219 188.609 40.6037C188.51 39.9881 188.521 39.3308 188.644 38.6319C188.765 37.933 188.971 37.2835 189.261 36.6832C189.551 36.0829 189.902 35.5662 190.313 35.1333C190.725 34.7001 191.175 34.4306 191.663 34.3239C191.48 35.0989 191.419 35.9007 191.48 36.7286C191.541 37.5568 191.739 38.3355 192.075 39.0648C192.288 39.506 192.544 39.9082 192.841 40.2729C193.139 40.6378 193.501 40.9492 193.928 41.2075C193.806 41.466 193.626 41.7204 193.39 41.9711ZM218.702 37.6519C218.747 37.3026 218.816 36.9336 218.908 36.5462C218.999 36.159 219.114 35.7828 219.251 35.4181C219.388 35.0532 219.548 34.738 219.731 34.4723C219.914 34.2065 220.108 34.0163 220.314 33.9024C220.52 33.7884 220.73 33.7997 220.943 33.9365C221.172 34.0735 221.313 34.3621 221.367 34.8025C221.42 35.2435 221.367 35.7142 221.207 36.2159C221.046 36.7173 220.761 37.1884 220.349 37.6288C219.937 38.07 219.38 38.3583 218.679 38.4951C218.648 38.2826 218.656 38.0015 218.702 37.6519ZM227.921 37.6519C227.966 37.3026 228.035 36.9336 228.126 36.5462C228.218 36.159 228.332 35.7828 228.47 35.4181C228.607 35.0532 228.767 34.738 228.95 34.4723C229.133 34.2065 229.328 34.0163 229.533 33.9024C229.739 33.7884 229.949 33.7997 230.162 33.9365C230.391 34.0735 230.532 34.3621 230.586 34.8025C230.639 35.2435 230.586 35.7142 230.425 36.2159C230.265 36.7173 229.979 37.1884 229.568 37.6288C229.156 38.07 228.599 38.3583 227.898 38.4951C227.867 38.2826 227.875 38.0015 227.921 37.6519ZM236.488 38.9852C236.312 38.7955 236.099 38.6625 235.847 38.5862C235.595 38.5104 235.355 38.5029 235.126 38.5636C234.897 38.6244 234.752 38.784 234.692 39.0422C234.57 39.5286 234.375 40.0262 234.108 40.5349C233.841 41.0444 233.514 41.5267 233.125 41.9824C232.736 42.4381 232.297 42.8412 231.81 43.1905C231.321 43.5401 230.81 43.7908 230.277 43.9423C229.743 44.1101 229.301 44.1289 228.95 43.9996C228.599 43.8706 228.321 43.6503 228.115 43.3389C227.909 43.0271 227.761 42.6512 227.669 42.2103C227.578 41.7699 227.524 41.3142 227.509 40.8428C228.378 40.9038 229.152 40.7483 229.831 40.3755C230.509 40.0034 231.085 39.5092 231.558 38.8939C232.031 38.2788 232.389 37.5874 232.633 36.82C232.877 36.0526 233.014 35.2892 233.045 34.5293C233.06 33.815 232.953 33.211 232.724 32.7171C232.496 32.2235 232.187 31.8395 231.798 31.5662C231.409 31.2924 230.963 31.133 230.46 31.0874C229.957 31.0417 229.445 31.1105 228.927 31.2924C228.302 31.5055 227.772 31.851 227.338 32.3296C226.903 32.8085 226.54 33.3634 226.251 33.9934C225.961 34.6244 225.732 35.3039 225.564 36.0335C225.396 36.7627 225.274 37.481 225.199 38.1874C225.124 38.873 225.084 39.5292 225.075 40.1572C225.017 40.2824 224.956 40.4082 224.889 40.5349C224.622 41.0444 224.295 41.5267 223.906 41.9824C223.517 42.4381 223.078 42.8412 222.591 43.1905C222.102 43.5401 221.592 43.7908 221.058 43.9423C220.524 44.1101 220.082 44.1289 219.731 43.9996C219.38 43.8706 219.102 43.6503 218.896 43.3389C218.691 43.0271 218.542 42.6512 218.45 42.2103C218.359 41.7699 218.305 41.3142 218.29 40.8428C219.159 40.9038 219.933 40.7483 220.612 40.3755C221.29 40.0034 221.866 39.5092 222.339 38.8939C222.811 38.2788 223.17 37.5874 223.414 36.82C223.658 36.0526 223.795 35.2892 223.826 34.5293C223.841 33.815 223.734 33.211 223.506 32.7171C223.277 32.2235 222.968 31.8395 222.579 31.5662C222.19 31.2924 221.744 31.133 221.241 31.0874C220.738 31.0417 220.227 31.1105 219.708 31.2924C219.083 31.5055 218.553 31.851 218.119 32.3296C217.684 32.8085 217.321 33.3634 217.032 33.9934C216.742 34.6244 216.513 35.3039 216.346 36.0335C216.178 36.7627 216.056 37.481 215.98 38.1874C215.936 38.5859 215.907 38.9722 215.886 39.3516C215.739 39.4765 215.595 39.6023 215.442 39.7258C214.916 40.1514 214.363 40.5349 213.784 40.8769C213.204 41.219 212.601 41.5001 211.977 41.7204C211.351 41.9408 210.71 42.0738 210.055 42.1192L211.473 26.9847C211.565 26.6655 211.519 26.3847 211.336 26.1415C211.153 25.8983 210.916 25.7312 210.627 25.6401C210.337 25.5488 210.028 25.5566 209.7 25.6627C209.372 25.7694 209.102 26.0126 208.888 26.3919C208.781 26.9697 208.671 27.7597 208.557 28.7625C208.442 29.7653 208.328 30.8595 208.213 32.0448C208.099 33.23 207.985 34.4532 207.87 35.7142C207.756 36.9759 207.657 38.1533 207.573 39.2472C207.569 39.2958 207.566 39.3398 207.562 39.3878C207.429 39.5005 207.299 39.6142 207.161 39.7258C206.635 40.1514 206.082 40.5349 205.503 40.8769C204.923 41.219 204.321 41.5001 203.696 41.7204C203.07 41.9408 202.429 42.0738 201.774 42.1192L203.192 26.9847C203.284 26.6655 203.238 26.3847 203.055 26.1415C202.872 25.8983 202.635 25.7312 202.346 25.6401C202.056 25.5488 201.747 25.5566 201.419 25.6627C201.091 25.7694 200.821 26.0126 200.607 26.3919C200.501 26.9697 200.39 27.7597 200.276 28.7625C200.161 29.7653 200.047 30.8595 199.933 32.0448C199.818 33.23 199.704 34.4532 199.589 35.7142C199.475 36.9759 199.376 38.1533 199.292 39.2472C199.29 39.2692 199.289 39.2891 199.287 39.3111C199.048 39.4219 198.786 39.519 198.503 39.6006C198.213 39.6844 197.885 39.7339 197.519 39.7489C197.58 39.4751 197.63 39.1712 197.668 38.8369C197.706 38.5029 197.737 38.1533 197.76 37.7884C197.782 37.4241 197.79 37.0591 197.782 36.6945C197.774 36.3296 197.755 35.9956 197.725 35.6914C197.649 35.0385 197.508 34.4191 197.302 33.8338C197.096 33.2491 196.818 32.7593 196.467 32.3637C196.116 31.9687 195.678 31.7027 195.151 31.5662C194.626 31.4294 194.012 31.4748 193.31 31.7027C192.273 31.5662 191.339 31.6613 190.508 31.9878C189.677 32.3149 188.956 32.7894 188.346 33.4122C187.736 34.0357 187.237 34.7684 186.848 35.6119C186.459 36.4551 186.2 37.3214 186.07 38.21C186.015 38.5868 185.988 38.9618 185.98 39.336C185.744 39.8177 185.486 40.2388 185.201 40.5921C184.797 41.0935 184.377 41.5038 183.943 41.8228C183.508 42.142 183.077 42.3852 182.65 42.5523C182.223 42.7198 181.842 42.8337 181.507 42.8941C181.11 42.9702 180.729 42.978 180.363 42.917C179.997 42.8565 179.661 42.6816 179.357 42.3927C179.112 42.1802 178.925 41.8381 178.796 41.3671C178.666 40.896 178.59 40.3608 178.567 39.7602C178.544 39.1599 178.567 38.533 178.636 37.8798C178.705 37.2266 178.822 36.6072 178.99 36.0222C179.158 35.4372 179.371 34.913 179.631 34.4492C179.89 33.9862 180.195 33.6554 180.546 33.4579C180.744 33.4886 180.866 33.606 180.912 33.811C180.958 34.0163 180.969 34.2595 180.946 34.5405C180.923 34.8219 180.889 35.1105 180.843 35.4066C180.797 35.703 180.775 35.9502 180.775 36.1474C180.851 36.5577 180.999 36.877 181.221 37.1048C181.441 37.3327 181.69 37.466 181.964 37.5036C182.239 37.5417 182.509 37.4773 182.776 37.3098C183.043 37.143 183.26 36.877 183.428 36.512C183.443 36.5274 183.466 36.5349 183.497 36.5349L183.817 33.6404C183.909 33.2451 183.847 32.8958 183.634 32.5919C183.42 32.288 183.138 32.113 182.788 32.0676C182.345 31.4294 181.747 31.0914 180.992 31.0532C180.237 31.0154 179.463 31.2623 178.67 31.7941C178.182 32.144 177.751 32.626 177.378 33.2413C177.004 33.857 176.699 34.5405 176.463 35.2926C176.226 36.0448 176.058 36.8391 175.959 37.6748C175.86 38.5104 175.841 39.3236 175.902 40.1133C175.963 40.9038 176.104 41.6484 176.325 42.347C176.546 43.0462 176.855 43.6312 177.252 44.102C177.587 44.5123 177.968 44.8127 178.395 45.0027C178.822 45.1927 179.268 45.3101 179.734 45.3558C180.199 45.4012 180.66 45.3821 181.118 45.2988C181.575 45.2155 182.01 45.0978 182.421 44.9454C182.955 44.7482 183.505 44.4972 184.069 44.1933C184.633 43.8897 185.174 43.5248 185.693 43.0991C185.966 42.8753 186.228 42.6313 186.482 42.3696C186.598 42.6553 186.727 42.9317 186.882 43.1905C187.294 43.8741 187.85 44.429 188.552 44.8544C189.253 45.2797 190.115 45.4844 191.137 45.4697C192.235 45.4544 193.249 45.1774 194.18 44.6378C195.11 44.0988 195.872 43.3042 196.467 42.256C197.358 42.256 198.234 42.1096 199.096 41.819C199.089 41.911 199.081 42.0079 199.075 42.0966C199.014 42.9019 198.983 43.4487 198.983 43.7376C198.968 44.239 198.934 44.8581 198.88 45.5949C198.827 46.332 198.793 47.1069 198.778 47.9198C198.763 48.7326 198.793 49.5532 198.869 50.3817C198.945 51.2096 199.105 51.962 199.349 52.6383C199.593 53.3141 199.94 53.8878 200.39 54.3591C200.84 54.8299 201.431 55.1112 202.163 55.2023C202.941 55.3084 203.612 55.1717 204.176 54.792C204.74 54.412 205.198 53.8918 205.549 53.2308C205.899 52.5695 206.147 51.8061 206.292 50.9401C206.437 50.074 206.479 49.2039 206.418 48.3301C206.357 47.4562 206.196 46.6321 205.937 45.8575C205.678 45.0822 205.319 44.444 204.862 43.9423C205.137 43.8669 205.465 43.7226 205.846 43.5095C206.227 43.2969 206.62 43.0575 207.024 42.7915C207.123 42.7261 207.221 42.6573 207.32 42.5902C207.283 43.1286 207.264 43.5126 207.264 43.7376C207.249 44.239 207.215 44.8581 207.161 45.5949C207.108 46.332 207.073 47.1069 207.058 47.9198C207.043 48.7326 207.073 49.5532 207.15 50.3817C207.226 51.2096 207.386 51.962 207.63 52.6383C207.874 53.3141 208.221 53.8878 208.671 54.3591C209.121 54.8299 209.712 55.1112 210.444 55.2023C211.221 55.3084 211.892 55.1717 212.457 54.792C213.021 54.412 213.478 53.8918 213.83 53.2308C214.18 52.5695 214.428 51.8061 214.573 50.9401C214.718 50.074 214.759 49.2039 214.699 48.3301C214.637 47.4562 214.477 46.6321 214.218 45.8575C213.959 45.0822 213.601 44.444 213.143 43.9423C213.418 43.8669 213.745 43.7226 214.127 43.5095C214.508 43.2969 214.9 43.0575 215.305 42.7915C215.515 42.6533 215.724 42.5107 215.932 42.3641C216.01 43.1072 216.179 43.759 216.448 44.3073C216.776 44.9761 217.222 45.4925 217.787 45.8575C218.351 46.2218 219.014 46.4234 219.777 46.4612C220.539 46.4988 221.37 46.3586 222.271 46.0393C222.941 45.7965 223.525 45.4925 224.02 45.1279C224.516 44.763 224.962 44.3185 225.358 43.7946C225.381 43.7642 225.403 43.7313 225.425 43.7006C225.496 43.9134 225.574 44.1179 225.667 44.3073C225.995 44.9761 226.441 45.4925 227.006 45.8575C227.569 46.2218 228.233 46.4234 228.996 46.4612C229.758 46.4988 230.589 46.3586 231.489 46.0393C232.16 45.7965 232.744 45.4925 233.239 45.1279C233.735 44.763 234.181 44.3185 234.577 43.7946C234.974 43.27 235.336 42.666 235.664 41.9824C235.992 41.2985 236.323 40.5164 236.659 39.6347C236.72 39.3918 236.663 39.1752 236.488 38.9852Z" fill="#0D0C23"/>
</svg>`;
const paypal = `
<svg xmlns="http://www.w3.org/2000/svg" width="150" height="40">
<path fill="#253B80" d="M46.211 6.749h-6.839a.95.95 0 0 0-.939.802l-2.766 17.537a.57.57 0 0 0 .564.658h3.265a.95.95 0 0 0 .939-.803l.746-4.73a.95.95 0 0 1 .938-.803h2.165c4.505 0 7.105-2.18 7.784-6.5.306-1.89.013-3.375-.872-4.415-.972-1.142-2.696-1.746-4.985-1.746zM47 13.154c-.374 2.454-2.249 2.454-4.062 2.454h-1.032l.724-4.583a.57.57 0 0 1 .563-.481h.473c1.235 0 2.4 0 3.002.704.359.42.469 1.044.332 1.906zM66.654 13.075h-3.275a.57.57 0 0 0-.563.481l-.145.916-.229-.332c-.709-1.029-2.29-1.373-3.868-1.373-3.619 0-6.71 2.741-7.312 6.586-.313 1.918.132 3.752 1.22 5.031.998 1.176 2.426 1.666 4.125 1.666 2.916 0 4.533-1.875 4.533-1.875l-.146.91a.57.57 0 0 0 .562.66h2.95a.95.95 0 0 0 .939-.803l1.77-11.209a.568.568 0 0 0-.561-.658zm-4.565 6.374c-.316 1.871-1.801 3.127-3.695 3.127-.951 0-1.711-.305-2.199-.883-.484-.574-.668-1.391-.514-2.301.295-1.855 1.805-3.152 3.67-3.152.93 0 1.686.309 2.184.892.499.589.697 1.411.554 2.317zM84.096 13.075h-3.291a.954.954 0 0 0-.787.417l-4.539 6.686-1.924-6.425a.953.953 0 0 0-.912-.678h-3.234a.57.57 0 0 0-.541.754l3.625 10.638-3.408 4.811a.57.57 0 0 0 .465.9h3.287a.949.949 0 0 0 .781-.408l10.946-15.8a.57.57 0 0 0-.468-.895z"/>
<path fill="#179BD7" d="M94.992 6.749h-6.84a.95.95 0 0 0-.938.802l-2.766 17.537a.569.569 0 0 0 .562.658h3.51a.665.665 0 0 0 .656-.562l.785-4.971a.95.95 0 0 1 .938-.803h2.164c4.506 0 7.105-2.18 7.785-6.5.307-1.89.012-3.375-.873-4.415-.971-1.142-2.694-1.746-4.983-1.746zm.789 6.405c-.373 2.454-2.248 2.454-4.062 2.454h-1.031l.725-4.583a.568.568 0 0 1 .562-.481h.473c1.234 0 2.4 0 3.002.704.359.42.468 1.044.331 1.906zM115.434 13.075h-3.273a.567.567 0 0 0-.562.481l-.145.916-.23-.332c-.709-1.029-2.289-1.373-3.867-1.373-3.619 0-6.709 2.741-7.311 6.586-.312 1.918.131 3.752 1.219 5.031 1 1.176 2.426 1.666 4.125 1.666 2.916 0 4.533-1.875 4.533-1.875l-.146.91a.57.57 0 0 0 .564.66h2.949a.95.95 0 0 0 .938-.803l1.771-11.209a.571.571 0 0 0-.565-.658zm-4.565 6.374c-.314 1.871-1.801 3.127-3.695 3.127-.949 0-1.711-.305-2.199-.883-.484-.574-.666-1.391-.514-2.301.297-1.855 1.805-3.152 3.67-3.152.93 0 1.686.309 2.184.892.501.589.699 1.411.554 2.317zM119.295 7.23l-2.807 17.858a.569.569 0 0 0 .562.658h2.822c.469 0 .867-.34.939-.803l2.768-17.536a.57.57 0 0 0-.562-.659h-3.16a.571.571 0 0 0-.562.482z"/>
<path fill="#253B80" d="M7.266 29.154l.523-3.322-1.165-.027H1.061L4.927 1.292a.316.316 0 0 1 .314-.268h9.38c3.114 0 5.263.648 6.385 1.927.526.6.861 1.227 1.023 1.917.17.724.173 1.589.007 2.644l-.012.077v.676l.526.298a3.69 3.69 0 0 1 1.065.812c.45.513.741 1.165.864 1.938.127.795.085 1.741-.123 2.812-.24 1.232-.628 2.305-1.152 3.183a6.547 6.547 0 0 1-1.825 2c-.696.494-1.523.869-2.458 1.109-.906.236-1.939.355-3.072.355h-.73c-.522 0-1.029.188-1.427.525a2.21 2.21 0 0 0-.744 1.328l-.055.299-.924 5.855-.042.215c-.011.068-.03.102-.058.125a.155.155 0 0 1-.096.035H7.266z"/>
<path fill="#179BD7" d="M23.048 7.667c-.028.179-.06.362-.096.55-1.237 6.351-5.469 8.545-10.874 8.545H9.326c-.661 0-1.218.48-1.321 1.132L6.596 26.83l-.399 2.533a.704.704 0 0 0 .695.814h4.881c.578 0 1.069-.42 1.16-.99l.048-.248.919-5.832.059-.32c.09-.572.582-.992 1.16-.992h.73c4.729 0 8.431-1.92 9.513-7.476.452-2.321.218-4.259-.978-5.622a4.667 4.667 0 0 0-1.336-1.03z"/>
<path fill="#222D65" d="M21.754 7.151a9.757 9.757 0 0 0-1.203-.267 15.284 15.284 0 0 0-2.426-.177h-7.352a1.172 1.172 0 0 0-1.159.992L8.05 17.605l-.045.289a1.336 1.336 0 0 1 1.321-1.132h2.752c5.405 0 9.637-2.195 10.874-8.545.037-.188.068-.371.096-.55a6.594 6.594 0 0 0-1.017-.429 9.045 9.045 0 0 0-.277-.087z"/>
<path fill="#253B80" d="M9.614 7.699a1.169 1.169 0 0 1 1.159-.991h7.352c.871 0 1.684.057 2.426.177a9.757 9.757 0 0 1 1.481.353c.365.121.704.264 1.017.429.368-2.347-.003-3.945-1.272-5.392C20.378.682 17.853 0 14.622 0h-9.38c-.66 0-1.223.48-1.325 1.133L.01 25.898a.806.806 0 0 0 .795.932h5.791l1.454-9.225 1.564-9.906z"/>
</svg>`;

module.exports = RecentFilesPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
